<?php

namespace App\Http\Controllers\User;

use App\Helpers\ResponseData;
use App\Http\Controllers\Controller;
use App\Http\Resources\Order\OrderResource;
use App\Http\Resources\OrderResult\OrderResultResource;
use App\Models\Address;
use App\Models\Discount;
use App\Models\Order;
use App\Models\OrderQuestionAnswers;
use App\Models\OrderResult;
use App\Models\Service;
use App\Services\NotificationService\NotificationService;
use App\Services\OrderStatusService\OrderStatusService;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller
{
    public function index(Request $req)
    {
        $user = auth()->guard('users')->user();

        $orders = Order::query()->where('user_id', $user->id);
        if ($req->query('status', null) != null) {
            $status = $req->query('status');
            if ($status == "active") {
                $orders->whereIn('status', ['searching-expert', 'doing', 'expert-violation']);
            } else if ($status == "past") {
                $orders->whereIn('status', ['user-confirm','done']);
            } else if ($status == "canceled") {
                $orders->whereIn('status', ['canceled']);
            }
        }
        $orders = $orders->orderByDesc('updated_at')->paginate(10);

        return OrderResource::collection($orders);
    }

    public function details(Request $req)
    {
        $user = auth()->guard('users')->user();
        $order = Order::where('user_id', $user->id)->where('id', $req->id)->first();

        return ResponseData::success('order details', new OrderResource($order));
    }

    public function cancelOrder(Request $req)
    {
        $user = auth()->guard('users')->user();
        $order = Order::where('user_id', $user->id)->where('id', $req->id)->first();
        $order->status = 'canceled';
        $order->reason = $req->reason;
        $order->canceled_at = Carbon::now();
        $order->save();
        OrderStatusService::setStatus($order->id, 'canceled');

        return ResponseData::success('سفارش با موفقیت لغو شد');
    }

    public function create(Request $req)
    {
        $user = auth()->guard('users')->user();
        $date = $req->date;
        $time = $req->time;
        $service_id = $req->service;
        $address_id = $req->address;
        $discount_id = $req->discount_id;
        $answers = json_decode($req->answers, true);
        DB::beginTransaction();
        try {
            $address = Address::where('id', $address_id)->first();
            $service = Service::where('id', $service_id)->first();
            $addressString = null;
            $phoneString = null;
            $addressLat = null;
            $addressLang = null;

            if ($address) {
                $addressString = $address->address . ',پلاک' . $address->plate . ',واحد' . $address->unit;
                $phoneString = $address->phone;
                $addressLat = $address->lat;
                $addressLang = $address->lang;
            }

            $order = Order::create([
                'user_id' => $user->id,
                'service_id' => $service_id,
                'date' => $date,
                'time' => $time,
                'address' => $addressString,
                'phone' => $phoneString,
                'lat' => $addressLat,
                'lang' => $addressLang,
                'price' => $service->price_type =="agreement"?"agreement": $service->price,
                'discount_id' => $discount_id
            ]);
            foreach ($answers as $answer) {
                OrderQuestionAnswers::create([
                    'order_id' => $order->id,
                    'question_id' => $answer['question_id'],
                    'value' => $answer['value']['value'],
                    'extra' => $answer['value']['extra'],
                ]);
            }
            OrderStatusService::setStatus($order->id, 'searching-expert');
            NotificationService::sendNotification("user", $user->id, null, "expert", $order->id, null, "new-order");

            DB::commit();
            return ResponseData::success('سفارش با موفقیت ثبت شد');
        } catch (Exception $e) {
            DB::rollBack();
            return ResponseData::error('خطا در ثبت سفارش');
        }
    }

    public function getFactor(Request $req)
    {
        $result = OrderResult::where('order_id', $req->id)->first();
        if (!$result) {
            return ResponseData::error('فاکتور برای این سفارش ایجاد نشده است');
        }
        return ResponseData::success('Order Factor', new OrderResultResource($result));
    }

    public function manageOrderResult(Request $req)
    {
        $id = $req->id;
        $status = $req->status;
        $reason = $req->reason;

        try {
            $order = Order::where('id', $id)->first();
            if ($status == 'user-rejected') {
                OrderStatusService::setStatus($order->id, 'expert-violation');
                $order->status = 'expert-violation';
            } else {
                OrderStatusService::setStatus($order->id, 'user-confirm');
                $order->status = 'user-confirm';
                $order->completed_at = Carbon::now();

                //Add Dept
                $this->addExpertDebt($order);
            }
            $order->save();

            OrderResult::where('order_id', $id)->update(['status' => $status, 'reason' => $reason]);

            $type = "reject-result";
            if ($status != "user-rejected") {
                $type = "accept-result";
            }
            NotificationService::sendNotification(
                "user",
                $order->user_id,
                "",
                'expert',
                $order->id,
                null,
                $type
            );

            return ResponseData::success('عملیات با موفقیت انجام شد');
        } catch (Exception $e) {
            return ResponseData::error('عملیات انجام نشد');
        }
    }

    public function validateDiscount(Request $req)
    {
        $discount = Discount::where('code', $req->code)->first();
        if (!$discount) {
            return ResponseData::error('کد تخفیف نامعتبر است', null, 404);
        }

        return ResponseData::success('discount', $discount);
    }

}
